<?php

namespace App\Http\Controllers\Karyawan;

use Exception;
use Carbon\Carbon;
use App\Models\Shift;
use App\Models\Karyawan;
use App\Models\AjukanShift;
use App\Models\JadwalShift;
use App\Models\Departemen;
use App\Models\Notifikasi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class AjukanShiftController extends Controller
{
    /**
     * Middleware untuk memastikan hanya admin yang bisa akses
     */
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $user = Auth::user();

            // Cek apakah user memiliki role admin (admin, manager, gm, hrd)
            $adminRoles = ['admin', 'manager', 'gm', 'hrd'];
            if (!in_array($user->role, $adminRoles)) {
                return redirect()->route('karyawan.dashboard')
                    ->with('error', 'Anda tidak memiliki akses ke fitur ini.');
            }

            // Cek apakah user memiliki data karyawan dengan departemen
            if (!$user->karyawan || !$user->karyawan->departemen_id) {
                return redirect()->route('karyawan.dashboard')
                    ->with('error', 'Data departemen tidak ditemukan.');
            }

            return $next($request);
        });
    }

    /**
     * Display index page with shift info and pengajuan
     */
    public function index()
    {
        $user = Auth::user();
        $karyawan = $user->karyawan;
        $departemen = $karyawan->departemen;

        // Shift aktif departemen
        $jadwalShiftAktif = JadwalShift::where('departemen_id', $departemen->id)
            ->where('is_active', true)
            ->with('shift')
            ->first();

        // Pengajuan shift yang masih aktif (pending)
        $shiftAktif = AjukanShift::with(['shiftLama', 'shiftBaru', 'pemohon', 'approver'])
            ->where('departemen_id', $departemen->id)
            ->where('requested_by', $user->id)
            ->where('status', 'pending')
            ->latest()
            ->get();

        // Pengajuan shift yang sudah selesai
        $shiftSelesai = AjukanShift::with(['shiftLama', 'shiftBaru', 'pemohon', 'approver'])
            ->where('departemen_id', $departemen->id)
            ->where('requested_by', $user->id)
            ->whereIn('status', ['disetujui', 'ditolak'])
            ->latest()
            ->get();

        // Statistik
        $totalPengajuan = AjukanShift::where('departemen_id', $departemen->id)
            ->where('requested_by', $user->id)
            ->count();

        $pengajuanPending = AjukanShift::where('departemen_id', $departemen->id)
            ->where('requested_by', $user->id)
            ->where('status', 'pending')
            ->count();

        // Shift master untuk form
        $allShifts = Shift::all();

        // Cek apakah ada pengajuan pending
        $hasPending = $pengajuanPending > 0;

        return view('karyawan.ajukan-shift.index', compact(
            'departemen',
            'jadwalShiftAktif',
            'allShifts',
            'shiftAktif',
            'shiftSelesai',
            'totalPengajuan',
            'pengajuanPending',
            'hasPending'
        ));
    }

    /**
     * Show create form
     */
    public function create()
    {
        $user = Auth::user();
        $karyawan = $user->karyawan;
        $departemen = $karyawan->departemen;

        // Cek apakah ada pengajuan pending
        $hasPending = AjukanShift::where('departemen_id', $departemen->id)
            ->where('requested_by', $user->id)
            ->where('status', 'pending')
            ->exists();

        if ($hasPending) {
            return redirect()->route('karyawan.ajukan-shift.index')
                ->with('error', 'Anda masih memiliki pengajuan shift yang belum diproses.');
        }

        // Shift aktif saat ini
        $jadwalShiftAktif = JadwalShift::where('departemen_id', $departemen->id)
            ->where('is_active', true)
            ->with('shift')
            ->first();

        if (!$jadwalShiftAktif) {
            return redirect()->route('karyawan.ajukan-shift.index')
                ->with('error', 'Shift departemen Anda belum ditentukan.');
        }

        // Shift master (exclude shift aktif)
        $allShifts = Shift::where('id', '!=', $jadwalShiftAktif->shift_id)->get();

        return view('karyawan.ajukan-shift.create', compact(
            'departemen',
            'jadwalShiftAktif',
            'allShifts'
        ));
    }

    /**
     * Store new shift request
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'jenis' => 'required|in:sementara,permanen',
                'shift_baru_id' => 'required|exists:shift,id',
                'tanggal_mulai' => 'required|date|after_or_equal:today',
                'tanggal_selesai' => 'required_if:jenis,sementara|nullable|date|after_or_equal:tanggal_mulai',
                'alasan' => 'required|string|min:10|max:500'
            ]);

            $user = Auth::user();
            $karyawan = $user->karyawan;
            $departemen = $karyawan->departemen;

            // Cek apakah ada pengajuan yang masih pending
            $pendingExists = AjukanShift::where('departemen_id', $departemen->id)
                ->where('requested_by', $user->id)
                ->where('status', 'pending')
                ->exists();

            if ($pendingExists) {
                return back()->with('error', 'Masih ada pengajuan shift yang belum diproses.');
            }

            // Ambil shift aktif saat ini
            $jadwalShiftAktif = JadwalShift::where('departemen_id', $departemen->id)
                ->where('is_active', true)
                ->firstOrFail();

            $shiftLamaId = $jadwalShiftAktif->shift_id;
            $shiftBaruId = $request->shift_baru_id;

            // Validasi: shift baru tidak boleh sama dengan shift lama
            if ($shiftLamaId == $shiftBaruId) {
                return back()->with('error', 'Shift pengganti tidak boleh sama dengan shift saat ini!');
            }

            // Untuk shift permanen, tanggal_selesai = null
            $tanggalSelesai = $request->jenis == 'permanen' ? null : $request->tanggal_selesai;

            // Cek overlap untuk shift sementara yang sudah disetujui
            if ($request->jenis == 'sementara') {
                $overlap = AjukanShift::where('departemen_id', $departemen->id)
                    ->where('jenis', 'sementara')
                    ->where('status', 'disetujui')
                    ->where(function($query) use ($request) {
                        $query->whereBetween('tanggal_mulai', [$request->tanggal_mulai, $request->tanggal_selesai])
                            ->orWhereBetween('tanggal_selesai', [$request->tanggal_mulai, $request->tanggal_selesai])
                            ->orWhere(function($q) use ($request) {
                                $q->where('tanggal_mulai', '<=', $request->tanggal_mulai)
                                  ->where('tanggal_selesai', '>=', $request->tanggal_selesai);
                            });
                    })
                    ->exists();

                if ($overlap) {
                    return back()->with('error', 'Sudah ada pengajuan shift yang disetujui pada periode tersebut!');
                }
            }

            DB::beginTransaction();

            // Buat pengajuan
            $ajukanShift = AjukanShift::create([
                'departemen_id' => $departemen->id,
                'shift_lama_id' => $shiftLamaId,
                'shift_baru_id' => $shiftBaruId,
                'tanggal_mulai' => $request->tanggal_mulai,
                'tanggal_selesai' => $tanggalSelesai,
                'jenis' => $request->jenis,
                'requested_by' => $user->id,
                'alasan' => $request->alasan,
                'status' => 'pending'
            ]);

            // Activity log
            activity_log(
                'shift',
                'create',
                "Mengajukan pergantian shift {$request->jenis} untuk departemen {$departemen->nama}"
            );

            // Notifikasi untuk pemohon
            Notifikasi::create([
                'user_id' => $user->id,
                'judul' => 'Pengajuan Shift Berhasil',
                'pesan' => "Pengajuan pergantian shift {$request->jenis} Anda berhasil dikirim dan menunggu persetujuan admin.",
                'type' => 'shift',
                'target_role' => 'karyawan'
            ]);

            // Notifikasi untuk super admin
            Notifikasi::create([
                'user_id' => null,
                'judul' => 'Pengajuan Shift Baru',
                'pesan' => "{$user->nama} mengajukan pergantian shift {$request->jenis} untuk departemen {$departemen->nama}.",
                'type' => 'shift',
                'target_role' => 'super_admin'
            ]);

            DB::commit();

            $jenisText = $request->jenis == 'sementara' ? 'sementara' : 'permanen';

            return redirect()->route('karyawan.ajukan-shift.index')
                ->with('success', "Pengajuan shift {$jenisText} berhasil dikirim!");

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error Store Ajukan Shift: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Show riwayat page
     */
    public function riwayat()
    {
        $user = Auth::user();
        $karyawan = $user->karyawan;

        $pengajuan = AjukanShift::with(['shiftLama', 'shiftBaru', 'pemohon', 'approver', 'departemen'])
            ->where('departemen_id', $karyawan->departemen_id)
            ->where('requested_by', $user->id)
            ->latest()
            ->paginate(20);

        return view('karyawan.ajukan-shift.riwayat', compact('pengajuan'));
    }

    /**
     * Show detail pengajuan
     */
    public function show($id)
    {
        $user = Auth::user();

        $ajukan = AjukanShift::with(['shiftLama', 'shiftBaru', 'pemohon', 'approver', 'departemen'])
            ->where('requested_by', $user->id)
            ->findOrFail($id);

        return view('karyawan.ajukan-shift.detail', compact('ajukan'));
    }

    /**
     * Cancel pengajuan (hanya yang masih pending)
     */
    public function cancel($id)
    {
        try {
            $user = Auth::user();

            $ajukan = AjukanShift::where('id', $id)
                ->where('requested_by', $user->id)
                ->where('status', 'pending')
                ->firstOrFail();

            DB::transaction(function () use ($ajukan, $user) {
                $ajukan->update([
                    'status' => 'ditolak',
                    'catatan_admin' => 'Dibatalkan oleh pemohon'
                ]);

                activity_log(
                    'shift',
                    'cancel',
                    "Membatalkan pengajuan shift {$ajukan->jenis}"
                );
            });

            return back()->with('success', 'Pengajuan shift berhasil dibatalkan');

        } catch (Exception $e) {
            Log::error('Error Cancel Ajukan Shift: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }
}
